from flask import Flask, render_template, request, jsonify
from datetime import datetime, timedelta
from gplay_scraper import GPlayScraper

# NEW: imports for caching + parallelism
from functools import lru_cache
from concurrent.futures import ThreadPoolExecutor, as_completed

app = Flask(__name__, template_folder="templates", static_folder="static")
scraper = GPlayScraper(http_client="curl_cffi")  # keeps your reliability setup

# ---------- helpers ---------------------------------------------------------
def _num(val):
    try:
        if val is None:
            return None
        if isinstance(val, (int, float)):
            return val
        # "10,000,000+" -> 10000000
        return int(str(val).replace(",", "").replace("+", "").strip())
    except Exception:
        return None

def _parse_date(s):
    # gplay-scraper often returns "Oct 2, 2024" or ISO strings
    if not s:
        return None
    for fmt in ("%b %d, %Y", "%b %d, %y", "%Y-%m-%d", "%Y-%m-%dT%H:%M:%S", "%Y-%m-%dT%H:%M:%S.%f"):
        try:
            return datetime.strptime(s, fmt)
        except Exception:
            pass
    try:
        return datetime.fromisoformat(s)
    except Exception:
        return None

# NEW: cache the per-app fields (reduces repeated requests)
@lru_cache(maxsize=5000)
def _cached_fields(app_id, lang, country):
    try:
        return scraper.app_get_fields(
            app_id,
            [
                "title", "developer", "genre", "score",
                "installs", "minInstalls", "realInstalls",
                "dailyInstalls", "realDailyInstalls",
                "released", "appAgeDays", "lastUpdated",
                "free", "price"
            ],
            lang=lang, country=country
        )
    except Exception:
        return {}

# UPDATED: parallel enrichment + uses cache
def _enrich_apps_basic(apps, country="us", lang="en", limit=None, workers=8):
    """
    Pull per-app details for daily installs + release info.
    Parallelized and cached. Only enrich first `limit` apps.
    """
    out = []
    n = len(apps) if not limit else max(0, min(limit, len(apps)))
    ids = [a.get("appId") for a in apps[:n]]

    results = [None] * n
    if n > 0:
        with ThreadPoolExecutor(max_workers=workers) as pool:
            futs = {
                pool.submit(_cached_fields, app_id, lang, country): i
                for i, app_id in enumerate(ids) if app_id
            }
            for fut in as_completed(futs):
                i = futs[fut]
                fields = fut.result() or {}
                row = dict(apps[i] if i < len(apps) else {})
                row.update(fields)
                row["installsParsed"] = _num(row.get("realInstalls")) or _num(row.get("minInstalls")) or _num(row.get("installs"))
                row["dailyParsed"]     = _num(row.get("realDailyInstalls")) or _num(row.get("dailyInstalls"))
                row["released_dt"]     = _parse_date(row.get("released"))
                results[i] = row

        out.extend([r for r in results if r])

    # append untouched tail (if any)
    if n < len(apps):
        for a in apps[n:]:
            a = dict(a)
            a["installsParsed"] = _num(a.get("realInstalls")) or _num(a.get("minInstalls")) or _num(a.get("installs"))
            out.append(a)
    return out

def _format_chart_items(items):
    # add numeric rank if missing (list already ordered), and normalize
    for idx, a in enumerate(items, 1):
        a.setdefault("rank", idx)
        a["installsParsed"] = _num(a.get("realInstalls")) or _num(a.get("minInstalls")) or _num(a.get("installs"))
    return items

# ---------- routes ----------------------------------------------------------
@app.route("/")
def index():
    return render_template("index.html")

@app.route("/api/app")
def api_app():
    app_id = request.args.get("appId")
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")
    if not app_id:
        return jsonify(ok=False, error="Missing appId"), 400
    try:
        data = scraper.app_analyze(app_id, lang=lang, country=country)
        data["installsParsed"] = _num(data.get("realInstalls")) or _num(data.get("minInstalls")) or _num(data.get("installs"))
        data["dailyParsed"] = _num(data.get("realDailyInstalls")) or _num(data.get("dailyInstalls"))
        return jsonify(ok=True, data=data)
    except Exception as e:
        return jsonify(ok=False, error=str(e))

@app.route("/api/similar")
def api_similar():
    app_id = request.args.get("appId")
    count = int(request.args.get("count", 24))
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")
    try:
        items = scraper.similar_analyze(app_id, count=count, lang=lang, country=country)
        return jsonify(ok=True, items=_format_chart_items(items))
    except Exception as e:
        return jsonify(ok=False, error=str(e))

@app.route("/api/search")
def api_search():
    q = request.args.get("q", "").strip()
    count = int(request.args.get("count", 30))
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")
    enrich = request.args.get("enrich", "0") == "1"
    enrich_n = int(request.args.get("enrich_n", 20))

    if not q:
        return jsonify(ok=False, error="Empty query"), 400

    try:
        items = scraper.search_analyze(q, count=count, lang=lang, country=country)
        items = _format_chart_items(items)
        if enrich:
            items = _enrich_apps_basic(items, country=country, lang=lang, limit=enrich_n)
        return jsonify(ok=True, items=items)
    except Exception as e:
        return jsonify(ok=False, error=str(e))

@app.route("/api/developer")
def api_developer():
    dev_id = request.args.get("devId", "").strip()
    count = int(request.args.get("count", 60))
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")
    try:
        items = scraper.developer_analyze(dev_id, count=count, lang=lang, country=country)
        return jsonify(ok=True, items=_format_chart_items(items))
    except Exception as e:
        return jsonify(ok=False, error=str(e))

@app.route("/api/suggest")
def api_suggest():
    term = request.args.get("term", "").strip()
    count = int(request.args.get("count", 10))
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")
    try:
        items = scraper.suggest_analyze(term, count=count, lang=lang, country=country)
        return jsonify(ok=True, items=items)
    except Exception as e:
        return jsonify(ok=False, error=str(e))

@app.route("/api/suggest_nested")
def api_suggest_nested():
    term = request.args.get("term", "").strip()
    count = int(request.args.get("count", 5))
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")
    try:
        items = scraper.suggest_nested(term, count=count, lang=lang, country=country)
        return jsonify(ok=True, items=items)
    except Exception as e:
        return jsonify(ok=False, error=str(e))

# NEW: reviews endpoint
@app.route("/api/reviews")
def api_reviews():
    app_id = request.args.get("appId", "").strip()
    sort = request.args.get("sort", "NEWEST").upper()      # NEWEST | RELEVANT | RATING
    count = int(request.args.get("count", 20))
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")

    if not app_id:
        return jsonify(ok=False, error="Missing appId"), 400

    try:
        raw = scraper.reviews_analyze(app_id, sort=sort, count=count, lang=lang, country=country)
        # Normalize shape
        if isinstance(raw, list):
            items = raw
        elif isinstance(raw, dict):
            items = raw.get("items") or raw.get("reviews") or raw.get("data") or []
        else:
            items = []

        # keep only simple fields we render; avoid huge payloads
        slim = []
        for r in items:
            slim.append({
                "userName": r.get("userName") or r.get("user"),
                "score": r.get("score") or r.get("rating"),
                "text": r.get("text") or r.get("content"),
                "date": r.get("date") or r.get("at"),
            })
        return jsonify(ok=True, items=slim)
    except Exception as e:
        return jsonify(ok=False, error=str(e))

@app.route("/api/list")
def api_list():
    """
    Supports:
      - TOP_FREE, TOP_PAID, TOP_GROSSING (library native)
      - TOP_NEW_FREE (simulated), TOP_NEW_PAID (simulated), TRENDING (simulated)

    Extra query params:
      - new_days (for *_NEW_*), default 30
      - enrich/enrich_n (optional; TRENDING forces enrich)
    """
    collection = request.args.get("chart", "TOP_FREE").upper()
    category = request.args.get("category", "APPLICATION")
    count = int(request.args.get("count", 50))
    lang = request.args.get("lang", "en")
    country = request.args.get("country", "us")

    # optional enrichment flags
    enrich = request.args.get("enrich", "0") == "1"
    enrich_n = int(request.args.get("enrich_n", 30))

    # Simulated options need release/daily data, so we’ll enrich as required.
    simulated = collection in {"TOP_NEW_FREE", "TOP_NEW_PAID", "TRENDING"}
    if simulated:
        # Base list from a real chart first
        base_collection = "TOP_FREE" if collection == "TOP_NEW_FREE" else \
                          "TOP_PAID" if collection == "TOP_NEW_PAID" else "TOP_FREE"
        try:
            base = scraper.list_analyze(base_collection, category=category, count=count, lang=lang, country=country)
            base = _format_chart_items(base)
        except Exception as e:
            return jsonify(ok=False, error=str(e))

        # UPDATED: Always enrich for simulated charts, but respect user limit
        enriched = _enrich_apps_basic(
            base, country=country, lang=lang,
            limit=min(enrich_n, count),   # <= was max(), now min() for speed
            workers=8
        )

        if collection in {"TOP_NEW_FREE", "TOP_NEW_PAID"}:
            new_days = int(request.args.get("new_days", 30))
            cutoff = datetime.utcnow() - timedelta(days=new_days)

            def _is_ok(app):
                # free/paid filter
                if collection == "TOP_NEW_FREE" and not bool(app.get("free", True)):
                    return False
                if collection == "TOP_NEW_PAID" and bool(app.get("free", True)):
                    return False
                # release window
                dt = app.get("released_dt")
                if not dt:
                    # fallback to age days if provided
                    age = app.get("appAgeDays")
                    if age is not None:
                        try:
                            return int(age) <= new_days
                        except Exception:
                            pass
                    return False
                return dt >= cutoff

            filtered = [a for a in enriched if _is_ok(a)]
            # rank by installs (desc), tie-break by rating
            filtered.sort(key=lambda a: (_num(a.get("installsParsed")) or 0, a.get("score") or 0), reverse=True)
            # re-number ranks for clarity
            for i, a in enumerate(filtered, 1):
                a["rank"] = i
            return jsonify(ok=True, items=filtered[:count])

        elif collection == "TRENDING":
            # rank by daily installs (desc), fallback to 0
            enriched.sort(key=lambda a: (a.get("dailyParsed") or 0), reverse=True)
            for i, a in enumerate(enriched, 1):
                a["rank"] = i
            return jsonify(ok=True, items=enriched[:count])

    # Native charts: pass-through (with optional enrichment if user asked)
    try:
        items = scraper.list_analyze(collection, category=category, count=count, lang=lang, country=country)
        items = _format_chart_items(items)
        if enrich:
            items = _enrich_apps_basic(items, country=country, lang=lang, limit=enrich_n, workers=8)
        return jsonify(ok=True, items=items)
    except Exception as e:
        return jsonify(ok=False, error=str(e))

# ---------- main ------------------------------------------------------------
if __name__ == "__main__":
    app.run(host="127.0.0.1", port=5000, debug=True)
