(function () {
  const $  = (s)=>document.querySelector(s);
  const $$ = (s)=>Array.from(document.querySelectorAll(s));

  // --- utilities ------------------------------------------------------------
  const star = (x)=> x==null ? '—' : `⭐ ${Number(x).toFixed(1)}`;
  const pretty = (n)=>{
    if(n==null||isNaN(n)) return '—';
    if(n>=1e9) return (n/1e9).toFixed(1).replace(/\.0$/,'')+'B';
    if(n>=1e6) return (n/1e6).toFixed(1).replace(/\.0$/,'')+'M';
    if(n>=1e3) return (n/1e3).toFixed(1).replace(/\.0$/,'')+'K';
    return String(n);
  };
  const daily = (app)=> app.realDailyInstalls ?? app.dailyInstalls ?? null;

  const badge = (text, tone="slate") =>
    `<span class="text-xs px-2 py-1 rounded-full bg-${tone}-500/15 text-${tone}-300 border border-${tone}-500/20">${text}</span>`;

  // --- skeleton -------------------------------------------------------------
  const skeleton = () => `
    <div class="card p-4 animate-pulse">
      <div class="flex gap-3">
        <div class="w-14 h-14 rounded-lg bg-white/5"></div>
        <div class="flex-1 space-y-2">
          <div class="h-4 w-2/3 bg-white/5 rounded"></div>
          <div class="h-3 w-1/2 bg-white/5 rounded"></div>
          <div class="h-3 w-1/3 bg-white/5 rounded"></div>
        </div>
      </div>
      <div class="h-3 w-full bg-white/5 rounded mt-3"></div>
      <div class="h-3 w-3/4 bg-white/5 rounded mt-2"></div>
      <div class="h-8 w-40 bg-white/5 rounded mt-3"></div>
    </div>`;

  // --- card -----------------------------------------------------------------
  function appCard(app){
    const icon = app.icon || '';
    const installs = app.installsParsed ?? null;
    const d = daily(app);
    const rank = app.rank ? badge('#'+app.rank, 'emerald') : '';
    const rating = app.score ? badge(star(app.score), 'yellow') : badge('No rating', 'slate');
    const cat = app.genre || app.category || '';
    const total = installs != null ? badge(`Total ${pretty(installs)}`, 'sky') : '';
    const dailyB = d != null ? badge(`Daily ${pretty(d)}`, 'pink') : '';

    return `
      <div class="card p-4 shadow-soft hover:shadow-glow transition-shadow">
        <div class="flex gap-3">
          <img src="${icon}" class="w-14 h-14 rounded-lg object-cover bg-white/5" onerror="this.style.display='none'"/>
          <div class="min-w-0">
            <div class="font-semibold tracking-tight truncate">${app.title || 'Untitled'}</div>
            <div class="text-sm text-slate-400 truncate">${app.developer || ''}</div>
            <div class="mt-1 flex flex-wrap items-center gap-2">
              ${rank}${rating}${total}${dailyB}
            </div>
          </div>
        </div>

        <div class="mt-3 line-clamp-3 text-sm text-slate-200/90">${(app.summary||app.description||'').slice(0,180)}</div>

        <div class="mt-3 flex flex-wrap items-center gap-2">
          ${badge(cat || 'App', 'indigo')}
          <a href="https://play.google.com/store/apps/details?id=${app.appId}" target="_blank"
             class="px-3 py-1 rounded-lg bg-white text-base-900 text-sm font-medium hover:bg-slate-100">Play Store</a>
          <button class="px-3 py-1 rounded-lg bg-emerald-500 text-base-900 text-sm font-medium hover:bg-emerald-400"
                  data-appid="${app.appId}" data-action="details">Details</button>
          <button class="px-3 py-1 rounded-lg bg-sky-500 text-base-900 text-sm font-medium hover:bg-sky-400"
                  data-appid="${app.appId}" data-action="similar">Similar</button>
        </div>
      </div>`;
  }

  async function j(url){ try{ const r=await fetch(url); return await r.json(); }catch(e){ return {ok:false,error:String(e)} } }

  // --- tabs -----------------------------------------------------------------
  $$('.tab-btn').forEach(b=>b.addEventListener('click', ()=>{
    const t = b.dataset.tab;
    $$('.tab-btn').forEach(x=>x.setAttribute('aria-current', x===b ? 'true' : 'false'));
    ['charts','search','developer','suggest'].forEach(id=>$('#'+id).classList.add('hidden'));
    $('#'+t).classList.remove('hidden');
  }));

  // --- charts ---------------------------------------------------------------
  let lastChart = [];
  function sortChart(items){
    const mode = $('#chartSort').value;
    if(mode==='rank') items.sort((a,b)=>(a.rank||9999)-(b.rank||9999));
    else if(mode==='installs') items.sort((a,b)=>(b.installsParsed||0)-(a.installsParsed||0));
    else if(mode==='updated'){
      const tt = v => v ? (new Date(v)).getTime() : 0;
      items.sort((a,b)=> tt(b.updated)-tt(a.updated));
    }
  }

  $('#btnFetchChart').addEventListener('click', async ()=>{
    const params = new URLSearchParams({
      chart: $('#chart').value,
      category: $('#category').value,
      country: $('#country').value,
      lang: $('#lang').value,
      count: $('#count').value || 50,
      enrich: $('#enrich').checked ? '1' : '0',
      enrich_n: $('#enrich_n').value || '30'
    });

    const box = $('#cards');
    box.innerHTML = skeleton().repeat(6);

    const data = await j(`/api/list?${params}`);
    if(!data.ok){ box.innerHTML = `<div class="text-red-400">${data.error}</div>`; return; }

    lastChart = data.items || [];
    sortChart(lastChart);
    box.innerHTML = lastChart.map(appCard).join('');
  });

  $('#chartSort').addEventListener('change', ()=>{
    if(!lastChart.length) return;
    sortChart(lastChart);
    $('#cards').innerHTML = lastChart.map(appCard).join('');
  });

  // --- search ---------------------------------------------------------------
  $('#btnSearch').addEventListener('click', async ()=>{
    const q = $('#q').value.trim();
    const country = $('#s_country').value;
    const lang = $('#s_lang').value;
    const enrich = $('#s_enrich').checked ? '1' : '0';
    const enrich_n = $('#s_enrich_n').value || '20';
    const box = $('#searchCards');

    if(!q){ box.innerHTML = '<div class="text-slate-400">Type a keyword to search…</div>'; return; }
    box.innerHTML = skeleton().repeat(6);

    const data = await j(`/api/search?${new URLSearchParams({q, country, lang, count:30, enrich, enrich_n})}`);
    if(!data.ok){ box.innerHTML = `<div class="text-red-400">${data.error}</div>`; return; }

    let items = data.items || [];
    const sort = $('#searchSort').value;
    if(sort==='installs') items.sort((a,b)=>(b.installsParsed||0)-(a.installsParsed||0));
    else if(sort==='score') items.sort((a,b)=>(b.score||0)-(a.score||0));

    box.innerHTML = items.map(appCard).join('');
  });

  // --- developer ------------------------------------------------------------
  $('#btnDev').addEventListener('click', async ()=>{
    const devId = $('#devId').value.trim();
    const country = $('#d_country').value;
    const lang = $('#d_lang').value;
    const box = $('#devCards');

    if(!devId){ box.innerHTML = '<div class="text-slate-400">Enter a developer id…</div>'; return; }
    box.innerHTML = skeleton().repeat(6);

    const data = await j(`/api/developer?${new URLSearchParams({devId, country, lang, count:60})}`);
    if(!data.ok){ box.innerHTML = `<div class="text-red-400">${data.error}</div>`; return; }
    box.innerHTML = (data.items||[]).map(appCard).join('');
  });

  // --- suggest --------------------------------------------------------------
  $('#btnSuggest').addEventListener('click', async ()=>{
    const term = $('#sg_term').value.trim();
    const country = $('#sg_country').value;
    const lang = $('#sg_lang').value;
    const count = $('#sg_count').value || 10;
    const list = $('#sg_list');

    list.innerHTML = '<li class="text-slate-400">Loading…</li>';
    const data = await j(`/api/suggest?${new URLSearchParams({term,country,lang,count})}`);
    if(!data.ok){ list.innerHTML = `<li class="text-red-400">${data.error}</li>`; return; }
    list.innerHTML = (data.items||[]).map(s=>`<li class="px-3 py-2 rounded-lg bg-white/5 border border-white/10">${s}</li>`).join('') || '<li class="text-slate-400">No results.</li>';
  });

  $('#btnSuggestNested').addEventListener('click', async ()=>{
    const term = $('#sg_term').value.trim();
    const country = $('#sg_country').value;
    const lang = $('#sg_lang').value;
    const count = $('#sg_count').value || 5;
    const box = $('#sg_nested');

    box.innerHTML = '<div class="text-slate-400">Mining…</div>';
    const data = await j(`/api/suggest_nested?${new URLSearchParams({term,country,lang,count})}`);
    if(!data.ok){ box.innerHTML = `<div class="text-red-400">${data.error}</div>`; return; }
    const items = data.items || {};
    box.innerHTML = Object.keys(items).length
      ? Object.entries(items).map(([p,children])=>`
          <div class="card p-3">
            <div class="font-semibold mb-1">${p}</div>
            <ul class="list-disc list-inside text-slate-200">${(children||[]).map(c=>`<li>${c}</li>`).join('')}</ul>
          </div>`).join('')
      : '<div class="text-slate-400">No nested suggestions.</div>';
  });

  // --- modal ---------------------------------------------------------------
  function openModal(title, html){
    $('#modalTitle').textContent = title;
    $('#modalBody').innerHTML = html;
    const m=$('#modal'); m.classList.remove('hidden'); m.classList.add('flex');
  }
  $('#modalClose').addEventListener('click', ()=>{ const m=$('#modal'); m.classList.add('hidden'); m.classList.remove('flex'); });
  $('#modal').addEventListener('click', (e)=>{ if(e.target.id==='modal') $('#modalClose').click(); });

  ['cards','searchCards','devCards'].forEach(id=>{
    const root = document.getElementById(id);
    if(!root) return;
    root.addEventListener('click', async (e)=>{
      const btn = e.target.closest('button[data-action]');
      if(!btn) return;
      const appId = btn.dataset.appid;
      const action = btn.dataset.action;

      if(action==='details'){
        const data = await j(`/api/app?${new URLSearchParams({appId})}`);
        if(!data.ok) return openModal('Error', `<p class="text-red-400">${data.error}</p>`);
        const d = data.data;
        const dailyStr = d.realDailyInstalls ?? d.dailyInstalls ?? null;
        openModal(d.title || 'App', `
          <div class='grid md:grid-cols-3 gap-5'>
            <div>
              <img src='${d.icon||''}' class='w-28 h-28 rounded-xl object-cover bg-white/5'/>
              <div class='mt-3 space-y-1 text-sm'>
                <div class='text-slate-300'>${d.developer||''}</div>
                <div class='text-slate-300'>${star(d.score)} · ${d.genre||''}</div>
                <div class='text-slate-400'>Total installs: <b class="text-slate-200">${pretty(d.installsParsed)}</b></div>
                <div class='text-slate-400'>Daily installs: <b class="text-slate-200">${pretty(dailyStr)}</b></div>
                <div class='text-slate-400'>Updated: ${d.lastUpdated || d.updated || '—'}</div>
              </div>
              <div class="mt-3">
                <a class="inline-flex items-center gap-2 px-3 py-1.5 rounded-lg bg-white text-base-900 text-sm font-medium" 
                   href="https://play.google.com/store/apps/details?id=${d.appId}" target="_blank">Open in Play Store</a>
              </div>
            </div>
            <div class='md:col-span-2 space-y-4'>
              <div>
                <div class='font-semibold mb-1'>Description</div>
                <div class='text-sm text-slate-200/90'>${(d.description||'').slice(0,1600)}</div>
              </div>

              <div class='flex items-end gap-2'>
                <div>
                  <label class='text-xs text-slate-400'>Reviews sort</label>
                  <select id='rv_sort' class='bg-base-800 rounded px-2 py-1 text-sm border border-white/10'>
                    <option value='NEWEST'>NEWEST</option>
                    <option value='RELEVANT'>RELEVANT</option>
                    <option value='RATING'>RATING</option>
                  </select>
                </div>
                <div>
                  <label class='text-xs text-slate-400'>Count</label>
                  <input id='rv_count' type='number' value='20' min='1' max='200'
                         class='bg-base-800 rounded px-2 py-1 text-sm border border-white/10 w-20'/>
                </div>
                <button id='btnLoadReviews' data-appid='${d.appId}'
                        class='px-3 py-2 rounded-lg bg-base-800 border border-white/10 text-sm hover:bg-white/5'>Load reviews</button>
              </div>
              <div id='reviews' class="space-y-3"></div>
            </div>
          </div>`);
      } else if(action==='similar'){
        const data = await j(`/api/similar?${new URLSearchParams({appId, count:18})}`);
        if(!data.ok) return openModal('Similar Apps', `<p class="text-red-400">${data.error}</p>`);
        openModal('Similar Apps', `<div class="grid sm:grid-cols-2 lg:grid-cols-3 gap-4">${data.items.map(appCard).join('')}</div>`);
      }
    });
  });

  // --- reviews loader (delegated on modal) ---------------------------------
  document.getElementById('modal').addEventListener('click', async (e) => {
    const btn = e.target.closest('#btnLoadReviews');
    if (!btn) return;

    const appId = btn.dataset.appid;
    const sortEl = document.getElementById('rv_sort');
    const countEl = document.getElementById('rv_count');
    const box = document.getElementById('reviews');

    const sort = sortEl ? sortEl.value : 'NEWEST';
    const count = countEl ? (parseInt(countEl.value, 10) || 20) : 20;

    // use current global filters so content localizes
    const lang = ($('#lang') && $('#lang').value) || 'en';
    const country = ($('#country') && $('#country').value) || 'us';

    box.innerHTML = '<div class="text-slate-400">Loading reviews…</div>';

    const qs = new URLSearchParams({ appId, sort, count, lang, country });
    let data;
    try {
      const res = await fetch(`/api/reviews?${qs}`);
      data = await res.json();
    } catch (err) {
      data = { ok: false, error: String(err) };
    }

    if (!data.ok) {
      box.innerHTML = `<div class="text-red-400">${data.error || 'Failed to load reviews'}</div>`;
      return;
    }

    const items = data.items || [];
    if (!items.length) {
      box.innerHTML = '<div class="text-slate-400">No reviews found.</div>';
      return;
    }

    const row = (r) => `
      <div class="p-3 rounded-lg bg-white/5 border border-white/10">
        <div class="flex items-center gap-2 text-sm">
          <span class="font-medium">${r.userName || 'User'}</span>
          <span class="text-amber-300">⭐ ${Number(r.score || r.rating || 0).toFixed(1)}</span>
          <span class="text-slate-400">${r.date || r.at || ''}</span>
        </div>
        <div class="mt-1 text-slate-200">${(r.text || r.content || '').replace(/\n/g,'<br>')}</div>
      </div>`;
    box.innerHTML = items.map(row).join('');
  });

  // auto-load
  window.addEventListener('DOMContentLoaded', ()=> $('#btnFetchChart').click());
})();
